' Sunset puzzle as seen on YouTube channel "Mr Puzzle"
' Rev 1.0.0 William M Leue 5/16/2021
option default integer
option base 1

' Constants
const USIZE = 60
const FWSPACE = 4
const FHSPACE = 5
const FTHICK = 8
const NUM_PIECES = 10
const NUM_ROWS = 5
const NUM_COLS = 4
const DRAD = 0.7*USIZE\2
const NUM_COLORS = 3
const FCOLOR = RGB(80, 30, 10)
const PBCOLOR = RGB(100, 80, 50)

const UP = 128
const DOWN = 129
const LEFT = 130
const RIGHT = 131
const SPACE = 32
const ESCAPE = 27
const HOME = 134

' Globals
dim cx, cy
dim pieces(NUM_PIECES, 5)
dim board(NUM_ROWS, NUM_COLS)
dim colors(NUM_COLORS) = (RGB(BLACK), RGB(RED), RGB(YELLOW))
dim selected = 0
dim captured = 0
dim nmoves = 0

' Main Program
ReadPieces
ShowRules
selected = 1
DrawPuzzle
HandleUserEvents
end

' Read the pieces, their initial location, and spot colors
' Also init the board squares
sub ReadPieces
  local i, j, row, col, sr, er, sc, ec
  for row = 1 to NUM_ROWS
    for col = 1 to NUM_COLS
      board(row, col) = 0
    next col
  next row
  for i = 1 to NUM_PIECES
    for j = 1 to 5
      read pieces(i, j)
    next j
    sr = pieces(i, 1) : er = sr + pieces(i, 4)-1
    sc = pieces(i, 2) : ec = sc + pieces(i, 3)-1
    for row = sr to er
      for col = sc to ec
        board(row, col) = i
      next col
    next row
  next i
end sub

' Reset the puzzle to the initial position
sub NewGame
  cls
  Restore
  ReadPieces
  DrawPuzzle
  nmoves = 0
  selected = 1
end sub

' Draw the Puzzle
sub DrawPuzzle
  DrawFrame
  DrawKeyMap
  DrawPieces
end sub

' Draw the Frame of the Puzzle
sub DrawFrame
  local x, y, x1, y1, x2, y2
  text MM.HRES\2, 10, "The Sunset Puzzle", "CT", 5
  cx = MM.HRES\2 : cy = MM.VRES\2 - 40
  x = cx - 2*USIZE
  y = cy + USIZE\2 - 2.5*USIZE
  box x-FTHICK, y-FTHICK, FWSPACE*USIZE+2*FTHICK, FHSPACE*USIZE+2*FTHICK, FTHICK, FCOLOR, FCOLOR
  x1 = cx-USIZE : y1 = cy+3*USIZE+2 : x2 = cx+USIZE : y2 = y1
  line x1, y1, x2, y2, 3, RGB(BLUE)
end sub

' Draw a reminder of the keyboard commands
sub DrawKeyMap
  text MM.HRES\2, 530, "Arrow Keys to Navigate and Move Pieces", "CT"
  text MM.HRES\2, 545, "Spacebar to Capture or Free a Piece", "CT"
  text MM.HRES\2, 560, "Home key to restart the Puzzle", "CT"
  text MM.HRES\2, 575, "Escape key to Quit", "CT"
end sub

' Draw the pieces whatever their current location
sub DrawPieces
  local i, row, col, w, h, x, y, bx, by, c, ec
  for i = 1 to NUM_PIECES
    row = pieces(i, 1) : col = pieces(i, 2)
    w = pieces(i, 3)   : h = pieces(i, 4)
    c = pieces(i, 5)
    if i = selected then
      if captured then
        ec = RGB(GREEN)
      else
        ec = RGB(WHITE)
      end if
    else
      ec = RGB(BLACK)
    end if
    x = cx - 2*USIZE + (col-1)*USIZE
    y = cy - 2*USIZE + (row-1)*USIZE
    box x, y, w*USIZE, h*USIZE,, ec, PBCOLOR
    bx = x + w*USIZE\2
    by = y + h*USIZE\2
    circle bx, by, DRAD,,, colors(c), colors(c)
  next i    
end sub

' Handle all user interaction when the puzzle is being solved
sub HandleUserEvents
  local z$
  local cmd, row, col, p, move, old_row, old_col
  row = 1 : col = 1
  z$ = INKEY$
  p = selected
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    move = 0
    cmd = asc(z$)
    old_row = row : old_col = col
    select case cmd
      case UP
        if row > 1 then
          row = row-1
        else
          row = NUM_ROWS
        end if
        move = 1
      case DOWN
        if row < NUM_ROWS then
          row = row+1
        else
          row = 1
        end if
        move = 1
      case LEFT
        if col > 1 then
          col = col-1
        else
          col = NUM_COLS
        end if
        move = 1
      case RIGHT
        if col < NUM_COLS then
          col = col+1
        else
          col = 1
        end if
        move = 1
      case SPACE
        captured = 1 - captured
        move = 0
      case HOME
        NewGame
      case ESCAPE
        cls
        end
    end select
    if move then
      if captured then
        MaybeMovePiece p, cmd
      else
        p = NextPieceInDirection(p, cmd)
        selected = p
      end if
    end if
    DrawPuzzle
  loop
end sub

' Given the captured piece index and the direction cmd, try to move the piece
sub MaybeMovePiece p, cmd
  local prow, pcol, pw, ph
  if p = 0 then exit sub
  prow = pieces(p, 1) : pcol = pieces(p, 2)
  pw = pieces(p, 3)   : ph = pieces(p, 4)
  select case cmd
    case UP, DOWN
      if HasVMoveSpace(p, cmd) then
        VMovePiece p, cmd
        CheckWin
      end if
    case LEFT, RIGHT
      if HasHMoveSpace(P, cmd) then
        HMovePiece p, cmd
        CheckWin
      end if
  end select
end sub

' Check for space above or below a piece that
' the piece can move into. Returns 1 for yes
' and 0 for no.
' Parameters: p: piece index, d: direction command
function HasVMoveSpace(p, d)
  local ok, prow, pcol, pw, ph, row, col
  prow = pieces(p, 1) : pcol = pieces(p, 2)
  pw = pieces(p, 3) : ph = pieces(p, 4)
  ok = 1
  if d = UP then
    row = prow-1
  else
    row = prow+ph
  end if
  if row < 1 or row > NUM_ROWS then
    HasVMoveSpace = 0
    exit function
  end if
  for col = pcol to pcol+pw-1
    if board(row, col) <> 0 then ok = 0
  next col
  HasVMoveSpace = ok
end function

' Check for space left or right of a piece that
' the piece can move into. Returns 1 for yes
' and 0 for no.
' Parameters: p: piece index, d: direction command
function HasHMoveSpace(p, d)
  local prow, pcol, pw, ph, row, col
  prow = pieces(p, 1) : pcol = pieces(p, 2)
  pw = pieces(p, 3)   : ph = pieces(p, 4)
  ok = 1
  if d = LEFT then
    col = pcol-1
    if col < 1 or col+pw-1 > NUM_COLS then
      HasHMoveSpace = 0
      exit function
    end if
  else
    col = pcol+pw
    if col < 1 or col > NUM_COLS then
      HasHMoveSpace = 0
      exit function
    end if
    limcol = col+pw-1
  end if
  for row = prow to prow+ph-1  
    if board(row, col) <> 0 then ok = 0
  next row
  HasHMoveSpace = ok
end function

' Unconditionally move a piece in the vertical
' dimension and update the board.
sub VMovePiece p, cmd
  local col, prow, pcol, pw, ph
  prow = pieces(p, 1) : pcol = pieces(p, 2)
  pw = pieces(p, 3)   : ph = pieces(p, 4)
  select case cmd
    case UP
      pieces(p, 1) = prow-1
      for col = pcol to pcol+pw-1
        board(prow-1, col) = p
        board(prow+ph-1, col) = 0
      next col
    case DOWN
      pieces(p, 1) = prow+1
      for col = pcol to pcol+pw-1
        board(prow+ph, col) = p
        board(prow, col) = 0
      next col
  end select
end sub

' Unconditionally move a piece in the horizontal
' dimension and update the board.
sub HMovePiece p, cmd
  local prow, pcol, pw, ph, row, col
  prow = pieces(p, 1) : pcol = pieces(p, 2)
  pw = pieces(p, 3)   : ph = pieces(p, 4)
  select case cmd
    case LEFT
      pieces(p, 2) = pcol-1
      for row = prow to prow+ph-1
        board(row, pcol-1) = p
        board(row, pcol+pw-1) = 0
      next row
    case RIGHT
      pieces(p, 2) = pcol+1
      for row = prow to prow+ph-1
        board(row, pcol+pw) = p
        board(row, pcol) = 0
      next row
  end select
end sub

' Find the next piece in the indicated direction from the current piece,
' and return its index. If no piece in that direction, wrap around to
' start and find the first piece, which might be the same as the current
' piece. P is the current piece index, d is the direction command.
function NextPieceInDirection(p, d)
  local prow, pcol, pw, ph, crow, ccol, rstart, cstart, b, np, hit
  prow = pieces(p, 1) : pcol = pieces(p, 2)
  pw = pieces(p, 3) : ph = pieces(p, 4)
  np = p
  hit = 0
  select case d
    case UP
      if prow > 1 then
        rstart = prow-1
      else
        rstart = NUM_ROWS
      end if
      for crow = rstart to 1 step -1
        for ccol = pcol to pcol+pw-1
          b = board(crow, pcol)
          if b > 0 and b <> p then
            np = b
            hit = 1
            exit for
          end if
        next ccol
        if hit then exit for
      next crow
    case DOWN
      if prow < NUM_ROWS then
        rstart = prow+1
      else
        rstart = 1
      end if
      np = p
      for crow = rstart to NUM_ROWS
        for ccol = pcol to pcol+pw-1
          b = board(crow, ccol)
          if b > 0 and b <> p then
            np = b
            hit = 1
            exit for
          end if
        next ccol
        if hit then exit for
      next crow
    case LEFT
      if pcol > 1 then
        cstart = pcol-1
      else
        cstart = NUM_COLS
      end if
      for ccol = cstart to 1 step -1
        for crow = prow to prow+ph-1
          b = board(crow, ccol)
          if b > 0 and b <> p then
            np = b
            hit = 1
            exit for
          end if
        next crow
        if hit then exit for
      next ccol
    case RIGHT
      if pcol < NUM_COLS then
        cstart = pcol+1
      else
        cstart = 1
      end if
      for ccol = cstart to NUM_COLS
        for crow = prow to prow+ph-1
          b = board(crow, ccol)
          if b > 0 and b <> p then
            np = b
            hit = 1
            exit for
          end if
        next crow
        if hit then exit for
      next ccol
  end select
  NextPieceInDirection = np
end function

' Check for a Win, track number of moves, and notify user
sub CheckWin
  local win = 0
  inc nmoves
  text 600, 30, space$(22)
  text 600, 200, "Number of Moves: " + str$(nmoves)
  if board(5, 2) = 2 and board(5, 3) = 2 then
    win = 1
  end if
  if win then
    text MM.HRES\2, 570, "YOU WIN!!", "CB", 5, 2, RGB(RED)
    text MM.HRES\2, 599, "Press any key to Quit", "CB"
    z$ = INKEY$
    do
      z$ = INKEY$
    loop until z$ <> ""
    cls
    end
  end if
end sub

sub ShowRules
  cls
  text MM.HRES\2, 10, "The Sunset Puzzle", "CT", 5
  text 0, 80, ""
  print "The Sunset Puzzle is a sliding block puzzle similar to the classic '15' puzzle."
  print "The goal is to get the 'Sun' piece (the 2x2 block with the red circle) to the"
  print "bottom center of the board. On a real puzzle, there is a slot on the bottom side"
  print "that is wide enough to admit the Sun piece but none of the others, which are"
  print "thicker. (This game shows the winning location for the bottom of the Sun piece with a"
  print "blue line.) Pieces can only be moved left, right, up, and down and only when there"
  print "is a space to accomodate them. No piece can be rotated or flipped."
  print "In the optimum solution it still takes more than 100 moves to solve."
  print ""
  print "Use the arrow keys to navigate around the board. Your position is marked by a white"
  print "outline on the current block. However, you first have to 'capture' the marked block"
  print "by pressing the space bar before you can move it. A 'captured' block is marked by"
  print "a green outline. Once you have captured a block, use the arrow keys to move it to"
  print "a suitable empty space on the board. Press space again to release a captured block."
  print ""
  print "Press the Home key to restart the game."
  print "You can also press the Escape key any time to quit."
  print ""
  print "The Sunset Puzzle was shown and solved on YouTube on the 'Mr Puzzle' channel. He shows"
  print "a solution if you get stuck and need help."

  text MM.HRES\2, 550, "Press any key to start", "CT", 3
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$  <> ""
  cls
end sub

' Initial Pieces (row, col, width, height, color)
' Note that the 1x2 pieces come in 4 vertical and 1 horizontal orientation,
' but neither can be rotated. The other pieces are symmetrical.
data 1, 1, 1, 2, 1
data 1, 2, 2, 2, 2
data 1, 4, 1, 2, 1
data 3, 2, 2, 1, 1
data 4, 1, 1, 2, 1
data 4, 2, 1, 1, 3
data 4, 3, 1, 1, 3
data 4, 4, 1, 2, 1
data 5, 2, 1, 1, 3
data 5, 3, 1, 1, 3
